/* **********************************************************
 * Copyright 1998 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/

#ifndef _VNETINT_H
#define _VNETINT_H

#define INCLUDE_ALLOW_MODULE
#include "includeCheck.h"
#include "vnet.h"
#include "vm_oui.h"
#include "net.h"

#include <asm/page.h>

#ifdef skb_shinfo
#  define SKB_IS_CLONE_OF(clone, skb)	(  \
      skb_shinfo(clone) == skb_shinfo(skb) \
   )
#else
#  define SKB_IS_CLONE_OF(clone, skb)	(      \
      skb_datarefp(clone) == skb_datarefp(skb) \
   )
#endif
#define DEV_QUEUE_XMIT(skb, dev, pri)	(                 \
    (skb)->dev = (dev),                                   \
    (skb)->priority = (pri),                              \
    skb_reset_mac_header(skb),                         \
    skb_set_network_header(skb, sizeof (struct ethhdr)), \
    dev_queue_xmit(skb)                                   \
  )
#ifdef KERNEL_2_3_15
#  define dev_lock_list()		read_lock(&dev_base_lock)
#  define dev_unlock_list()		read_unlock(&dev_base_lock)
#  define DEV_GET(x)                    __dev_get_by_name(x)
#else
#  define DEV_GET(x)			dev_get(x)
#endif


/*
 * Various fields (including 'dead') of struct sock are replaced with the
 * 'flags' bitfield in 2.5.65 --hpreg
 */
#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 5, 65)
#   define SET_SK_DEAD(_sk, _val) (_sk)->dead = (_val)
#else
#   define SET_SK_DEAD(_sk, _val) sock_valbool_flag(_sk, SOCK_DEAD, _val)
#endif


#define INLINE inline

     
/* 
 * Logging
 */

#ifdef notdef
#ifdef VMX86_RELEASE
#define LOGLEVEL 0
#else
#define LOGLEVEL 1
#endif
#endif
     
#define LOGLEVEL 1

#if LOGLEVEL >= 0
#define LOG(level, args) ((void) (LOGLEVEL >= (level) ? (printk args) : 0))
#else
#define LOG(level, args)
#endif

#define MAX(_a, _b)   (((_a) > (_b)) ? (_a) : (_b))

/*
 * Ethernet
 */

#define MAC_EQ(_a, _b)         !memcmp((_a), (_b), ETH_ALEN)
#define SKB_2_DESTMAC(_skb)    (((struct ethhdr *)(_skb)->data)->h_dest)
#define SKB_2_SRCMAC(_skb)     (((struct ethhdr *)(_skb)->data)->h_source)
#define UP_AND_RUNNING(_flags) (((_flags) & (IFF_RUNNING|IFF_UP)) == \
				(IFF_RUNNING|IFF_UP))
#ifdef KERNEL_2_3_47
#define NETDEV_UP_AND_RUNNING(dev) ((((dev)->flags) & IFF_UP) && netif_running(dev))
#else
#ifdef KERNEL_2_3_43
#define NETDEV_UP_AND_RUNNING(dev) ((((dev)->flags) & IFF_UP) && test_bit(LINK_STATE_START, &(dev)->state))
#else
#define NETDEV_UP_AND_RUNNING(dev) UP_AND_RUNNING((dev)->flags)
#endif
#endif

/*
 * Misc defines 
 */

#define NULL_TERMINATE_STRING(a) (a)[sizeof (a) - 1] = '\0'

/*
 * Fundamental sizes
 */

#define VNET_NUM_VNETS         256
#define VNET_MAJOR_NUMBER      119

#define NUM_JACKS_PER_HUB      32
#define VNET_MAX_QLEN          128

#define VNET_NUM_IPBASED_MACS  64
#define VNET_MAX_JACK_NAME_LEN 16

#define VNET_LADRF_LEN         8

#if ( defined(IFNAMSIZ) && (IFNAMSIZ >= 16) )
#define VNET_NAME_LEN          IFNAMSIZ
#else
#define VNET_NAME_LEN          16
#endif

/*
 * Data structures
 */

typedef struct proc_dir_entry VNetProcEntry;

typedef struct VNetJack VNetJack;
typedef struct VNetPort VNetPort;

/*
 *  The jack is the basic mechanism for connecting to objects
 *  that send packet between them.
 */

extern struct semaphore vnetStructureSemaphore;

struct VNetJack {
   VNetJack      *peer;
   int            numPorts;
   char           name[VNET_MAX_JACK_NAME_LEN];
   void          *private;     // private field for containing object
   int            index;       // private field for containing object
   VNetProcEntry *procEntry;   // private field for containing object
   
   void         (*free)(VNetJack *this);
   void         (*rcv)(VNetJack *this, struct sk_buff *skb);
   Bool         (*cycleDetect)(VNetJack *this, int generation);
   void         (*portsChanged)(VNetJack *this);
   int          (*isBridged)(VNetJack *this);
};


/*
 *  The port is an extension of the jack. It has a user level
 *  interface and an ethernet address. There are 3 types of ports:
 *  userif, netif, and bridge.
 */

struct VNetPort {
   VNetJack    jack;     // must be first
   unsigned    id;
   uint32      flags;
   uint8       paddr[ETH_ALEN];
   uint8       ladrf[VNET_LADRF_LEN];
   
   VNetPort   *next;
   
   int       (*fileOpRead)(VNetPort *this, struct file *filp,
                           char *buf, size_t count);
   int       (*fileOpWrite)(VNetPort *this, struct file *filp,
                            const char *buf, size_t count);
   int       (*fileOpIoctl)(VNetPort *this, struct file *filp,
                            unsigned int iocmd, unsigned long ioarg);   
   int       (*fileOpPoll)(VNetPort *this, struct file *filp,
                           poll_table *wait);
};



/*
 *  Functions exported from vnet module
 */

VNetJack *VNetHub_AllocVnet(int hubNum);
VNetJack *VNetHub_AllocPvn(uint8 id[VNET_PVN_ID_LEN]);

int VNetConnect(VNetJack *jack1, VNetJack *jack2);

VNetJack *VNetDisconnect(VNetJack *jack);

void VNetSend(VNetJack *jack, struct sk_buff *skb);

int VNetProc_MakeEntry(VNetProcEntry *parent, char *name, int mode,
                       VNetProcEntry **ret);

void VNetProc_RemoveEntry(VNetProcEntry *node, VNetProcEntry *parent);

int VNetPrintJack(VNetJack *jack, char *buf);

int VNet_MakeMACAddress(VNetPort *port);

int VNetSetMACUnique(VNetPort *port, const uint8 mac[ETH_ALEN]);

     
/*
 *  Utility functions
 */

extern uint8 AllMultiFilter[VNET_LADRF_LEN];

Bool VNetPacketMatch(uint8 *destAddr, uint8 *ifAddr, uint8 *ladrf, uint32 flags);

Bool VNetCycleDetectIf(char *name, int generation);

void VNetIncrModCount(int delta);

int VNetPrintPort(VNetPort *port, char *buf);

int VNetSnprintf(char *str, size_t size, const char *format, ...);

/*
 *  Procfs file system
 */

extern int VNetProc_Init(void);

extern void VNetProc_Cleanup(void);


/*
 *----------------------------------------------------------------------
 *
 * VNetCycleDetect --
 *
 *      Perform the cycle detect alogorithm for this generation.
 *
 * Results: 
 *      TRUE if a cycle was detected, FALSE otherwise.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

static INLINE Bool
VNetCycleDetect(VNetJack *jack,       // IN: jack
                int       generation) // IN: 
{
   if (jack && jack->cycleDetect) {
      return jack->cycleDetect(jack, generation);
   }

   return FALSE;
}


/*
 *----------------------------------------------------------------------
 *
 * VNetPortsChanged --
 *
 *      Notify a jack that the number of connected ports has changed.
 *	vnetStructureSemaphore must be held.
 *
 * Results: 
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

static INLINE void
VNetPortsChanged(VNetJack *jack) // IN:
{
   if (jack && jack->portsChanged) {
      jack->portsChanged(jack);
   }
}


/*
 *----------------------------------------------------------------------
 *
 * VNetIsBridged --
 *
 *      Check whether we are bridged.
 *      vnetPeerLock must be held.
 *
 * Results:
 *      0 - not bridged
 *      1 - we are bridged but the interface is down
 *      2 - we are bridged and the interface is up
 *      3 - some bridges are down
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

static INLINE int
VNetIsBridged(VNetJack *jack) // IN: jack
{
   if (jack && jack->peer && jack->peer->isBridged) {
      return jack->peer->isBridged(jack->peer);
   }

   return 0;
}


/*
 *----------------------------------------------------------------------
 *
 * VNetFree --
 *
 *      Free the resources owned by the jack.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

static INLINE void
VNetFree(VNetJack *jack) // IN: jack
{
   if (jack && jack->free) {
      jack->free(jack);
   }
}


/*
 *----------------------------------------------------------------------
 *
 * VNetGetAttachedPorts --
 *
 *      Get the number of ports attached to this jack through its peer.
 *
 * Results:
 *      The number of attached ports to this jack through its peer.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

static INLINE int
VNetGetAttachedPorts(VNetJack *jack) // IN: jack
{
   if (jack && jack->peer) {
      return jack->peer->numPorts;
   }
   return 0;
}

#endif
